///////////////////////////////////////////////////////////////////////////////////////
//
// MonoComm
//
// Public MonoComm interface.
//
// Designed to be used with different Windows development tools (VC++, .NET, Delphi, VB, etc)
// Hence, a classic C-style iface with only Win32 types, no exceptions. You can create 
// your development tool specific header based on this one.
//
// When extending this interface to support other control units, add new functions whith a 
// prefix name and the required set of parameters instead of bloating the existing functions.
//
// 03/05/07		Hounddog	Sporadic fault attribute
// 25/02/07		Hounddog	Errors recovering
// 09/02/07		Hounddog	Initial implementation
//

#ifndef MONO_COMM_H
#define MONO_COMM_H

// MonoComm calling convention is STDCALL.

#define MC_API __stdcall

///////////////////////////////////////////////////////////////////////////////////////
// Error handling

typedef DWORD MC_STATUS;

static const MC_STATUS MC_S_OK							= 0;	// Success
static const MC_STATUS MC_S_ERROR						= -1;	// Error
static const MC_STATUS MC_S_ERROR_IMPL					= -2;	// Error, not implemented
static const MC_STATUS MC_S_ERROR_ECU					= -3;	// Error, ECU
static const MC_STATUS MC_S_ERROR_CONNECT				= -4;	// Error, connection
static const MC_STATUS MC_S_ERROR_CONNECT_PORT			= -5;	// Error, connection, comm port
static const MC_STATUS MC_S_ERROR_CONNECT_ECU			= -6;	// Error, connection, no ECU data
static const MC_STATUS MC_S_ERROR_CONNECT_ECU_ECHO		= -7;	// Error, connection, bad ECU echo
static const MC_STATUS MC_S_ERROR_CONNECT_ADAPTER		= -8;	// Error, connection, no adapter data
static const MC_STATUS MC_S_ERROR_CONNECT_BLOCK_INDEX	= -9;	// Error, connection, bad block index

///////////////////////////////////////////////////////////////////////////////////////
// Messaging

typedef DWORD MC_MESSAGE_LEVEL;

static const MC_MESSAGE_LEVEL MC_ML_NONE	= 0;
static const MC_MESSAGE_LEVEL MC_ML_ERROR	= 1;	// Most significant
static const MC_MESSAGE_LEVEL MC_ML_LOG_0	= 2;
static const MC_MESSAGE_LEVEL MC_ML_LOG_1	= 3;
static const MC_MESSAGE_LEVEL MC_ML_LOG_2	= 4;	// Less significant

typedef void (MC_API* MC_MESSAGE_HANDLER)(const VOID* pvContext, LPCSTR pszMessage);

///////////////////////////////////////////////////////////////////////////////////////
// Constructs MonoComm instance
//
// The instance is single-threaded.
//
// Currently only one MonoComm instance is supported at a time. Thus you cannot work in 
// several concurrent threads of the same process with several comm ports :)
//
// pvContext can be NULL
// mh can be NULL

MC_STATUS MC_API McConstruct(const VOID* pvContext, MC_MESSAGE_HANDLER mh);

///////////////////////////////////////////////////////////////////////////////////////
// Destructs MonoComm instance
//
// Shouldn't be called if McConstruct() failed.

void MC_API McDestruct();

///////////////////////////////////////////////////////////////////////////////////////
// Sets messages level. Messages with type > ml will not be generated.

void MC_API McSetMessagesLevel(MC_MESSAGE_LEVEL ml);

MC_MESSAGE_LEVEL MC_API McGetMessagesLevel();

///////////////////////////////////////////////////////////////////////////////////////
// MonoComm memory allocator

VOID* MC_API McAllocate(SIZE_T size);

void MC_API McFree(VOID* pv);

///////////////////////////////////////////////////////////////////////////////////////
// Connects to a control unit.
//
// dwBaudRate = 0 means automatic detection of control unit baud rate - not impled, you 
// must hint the control unit baud rate. Commonly used baud rates are 10400, 9600, 4800.
//
// bAddress is the identifier of the control unit within the car.
//
// pppszDescription is a NULL terminated array of '\0' terminated strings which identify
// control unit hardware and software, etc. The first string is control unit VAG part number.
// Each string and the array must be freed with McFree().
//
// MC_S_ERROR_CONNECT - the hinted baud rate is incorrect / something is wrong with unit.
// MC_S_ERROR_CONNECT_BLOCK_INDEX - something is wrong with the communication.
// MC_S_ERROR_CONNECT_ECU_ECHO - something is wrong with with the communication.
// MC_S_ERROR_CONNECT_ADAPTER_ECHO - something is wrong with the adapter.
// MC_S_ERROR_CONNECT_PORT - something is wrong with the serial communication port.
// McConnect() ensures a timeout on error to cancel the communication.

MC_STATUS MC_API McConnect(LPCSTR pszPortName, DWORD dwBaudRate, BYTE bAddress, LPSTR** pppszDescription);

///////////////////////////////////////////////////////////////////////////////////////
// Disconnects from control unit, either normally or via a timeout
//
// Shouldn't be called if McConnect() failed.

void MC_API McDisconnect();

///////////////////////////////////////////////////////////////////////////////////////
// McIdle() must be called at least once per each McGetMaxIdleInterval() to keep 
// the connection. If control unit doesn't see a block during McGetMaxIdleInterval() it 
// will assume the connection is lost and exit the communication.

DWORD MC_API McGetMaxIdleInterval();	// ms

MC_STATUS MC_API McIdle();

///////////////////////////////////////////////////////////////////////////////////////
// Retrieves control unit descriptions
//
// pppszDescription is a NULL terminated array of '\0' terminated strings. 
// Each string and the array must be freed with McFree().
//
// Actually you don't need to call this function at all, each time control unit returns 
// the same info as on the connection setup.

MC_STATUS MC_API McGetDescriptions(LPSTR** pppszDescription);

///////////////////////////////////////////////////////////////////////////////////////
// Retrieves faults (DTCs) from ECU
//
// If ECU returns a special fault indicating "no faults" you'll get it, not zero
// *pdwCount.
//
// ppfault must be freed with McFree().

struct MC_FAULT
{	
	BOOL bSporadic;
	WORD wCode;
	BYTE bSubCode;	
};

MC_STATUS MC_API McGetFaults(DWORD* pdwCount, struct MC_FAULT** ppfault);

///////////////////////////////////////////////////////////////////////////////////////
// Clears faults (DTCs) in ECU

MC_STATUS MC_API McClearFaults();

///////////////////////////////////////////////////////////////////////////////////////
// Retrieves basic settings from ECU
//
// ppbZone must be freed with McFree().
//
// Just call McGetBasicSettings() repeatedly with the desired interval <= McGetMaxIdleInterval()
// ECU automatically enters the basic settings mode upon recieving the first basic setting block 
// and automatically leaves the basic settings mode upon recieving an idle block.
//
// McGetBasicSettingsVar() allocates a small block of memory on each call. If you call it 
// frequently and long and do something in between it can fragment your memory and 
// slow down. On the other hand the number of zones is constant for a particular ECU.
// The idea is at first to call McGetBasicSettingsVar() and process the result, keep the allocated 
// buffer and call McGetBasicSettings() and process the result as many times as needed with the 
// count and buffer returned by McGetBasicSettingsVar() and free the buffer in the end.

MC_STATUS MC_API McGetBasicSettingsVar(DWORD* pdwCount, BYTE** ppbZone);

///////////////////////////////////////////////////////////////////////////////////////
// Retrieves basic settings from ECU
// 
// If the actual number of zones doesn't equal the number of zones requested, McGetBasicSettings()
// fails with MC_S_ERROR. Typically there're 10 zones.

MC_STATUS MC_API McGetBasicSettings(DWORD dwCount, BYTE* pbZone);

///////////////////////////////////////////////////////////////////////////////////////
// Starts testing of the next actuator.
//
// Returns the code of the started actuator, 0 if there're no more actuators to test

MC_STATUS MC_API McTestActuator(WORD* pdwActuatorCode);

#endif